unit fROR_ReportSelector;

interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, StdCtrls, Buttons, ExtCtrls, ovcstate, uROR_State, ovcbase,
  ovcfiler, ovcstore, uROR_VistAStore, ImgList, uROR_Broker, ActnList,
  ComCtrls, uROR_ListView, fROR_ReportParams, uROR_Selector,
  uROR_CustomControls, uROR_ParamsPanel, fROR_VistARegistries, uROR_Errors,
  uROR_Reports, o32editf, o32flxed, uROR_CustomListView,
  VA508AccessibilityManager;

type
  TFormReportSelector = class(TForm)
    pnlButtons: TPanel;
    pnlMain: TPanel;
    btnRun: TBitBtn;
    btnCancel: TBitBtn;
    pnlReportList: TPanel;
    splMain: TSplitter;
    pnlCurrentReport: TPanel;
    pnlReportName: TPanel;
    imlSort: TImageList;
    vxsFormLayout: TCCRVistAStore;
    vxsFormState: TCCRFormState;
    vxsComponentState: TCCRComponentState;
    ActionList: TActionList;
    acShowReportList: TAction;
    lvReports: TCCRListView;
    chkbShowReportList: TCheckBox;
    pnlReportParams: TPanel;
    Panel1: TPanel;
    pnlRightSpacer: TPanel;
    pnlBottomSpacer: TPanel;
    pnlLeftSpacer: TPanel;
    btnSaveParams: TBitBtn;
    btnLoadParams: TBitBtn;
    acLoadParams: TAction;
    acSaveParams: TAction;
    btnDefaultParams: TBitBtn;
    acDefaultParams: TAction;
    VA508AccessibilityManager1: TVA508AccessibilityManager;
    dlgOpen: TOpenDialog;
    acLoadTest: TAction;
    btnTestLoad: TBitBtn;
    acTestSave: TAction;
    btnTestSave: TBitBtn;
    spbUp: TSpeedButton;
    spbDown: TSpeedButton;
    acTestUp: TAction;
    acTestDown: TAction;
    Label1: TLabel;
    pnlDebug: TPanel;
    ckbDetails: TCheckBox;
    procedure vxsFormLayoutBeforeOpen(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure lvReportsSelectItem(Sender: TObject; Item: TListItem;
      Selected: Boolean);
    procedure vxsFormStateRestoreState(Sender: TObject);
    procedure vxsFormStateSaveState(Sender: TObject);
    procedure acShowReportListExecute(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure btnRunClick(Sender: TObject);
    procedure acSelectedUpdate(Sender: TObject);
    procedure acLoadParamsExecute(Sender: TObject);
    procedure acSaveParamsExecute(Sender: TObject);
    procedure lvReportsEnter(Sender: TObject);
    procedure acDefaultParamsExecute(Sender: TObject);
    procedure acLoadTestExecute(Sender: TObject);
    procedure acTestSaveExecute(Sender: TObject);
    procedure acTestUpExecute(Sender: TObject);
    procedure acTestDownExecute(Sender: TObject);

  private
    fParamsFrame:         TFrameReportParameters;
    fRegInfo:             TRegistryInfo;
    fDisplayHepCErrorMsg: boolean;
    fLastReportCode:      String;

    function getDateTimeToStart: String;
    function getReportDescriptor: TReportDescriptor;
    function getSchedule: String;
    function makeInstancePrefix(aReport: TReportDescriptor): String;
    procedure DisplayHepCErrorMessage(aRegistry, aRptCode: String);

  protected
    function SelectReport(const aReportCode: String): Boolean; virtual;

    property RegInfo: TRegistryInfo              read  fRegInfo;

  public
    function  Load(aRegInfo: TRegistryInfo; Reload: Boolean = False): TReturnCode;
    function  Open(aRegInfo: TRegistryInfo; const aReportCode: String = ''): Boolean;

    property DateTimeToStart: String             read getDateTimeToStart;
    property ParamsFrame: TFrameReportParameters read fParamsFrame;
    property ReportDescriptor: TReportDescriptor read getReportDescriptor;
    property Schedule: String                    read getSchedule;

  published
    procedure ApplyOptions;

  end;

function ReportSelector: TFormReportSelector;
function SetErrorCodeForLocalHepCTests: integer;

implementation
{$R *.dfm}

uses
  uROR_Common, uROR_CommonUtils, fROR_RPLoadSave, fROR_Options, uROR_Strings,
  uROR_Utilities, uROR_Classes, uROR_TemplateFormatFix, fROR_Info
{$IFDEF RPCLOG}
  , fZZ_EventLog
{$ENDIF}
  ;

var
  FormReportSelector: TFormReportSelector = nil;

function ReportSelector: TFormReportSelector;
begin
  if not Assigned(FormReportSelector) then
    FormReportSelector := TFormReportSelector.Create(Application);
  Result := FormReportSelector;
end;

function SetErrorCodeForLocalHepCTests: integer;
var
  rawData: TStringList;
begin
  Result := 0;
  rawData := TStringList.Create;
  try
    //--- Check for local HEPC Quant tests (code = 4)
    if Broker.CallProc(rpcListLocalLabTests, ['1', '4'], nil, rawData) then
    begin
      rawData.Delete(0);
      if rawData.Count = 0 then
      begin
        //--- Check for local HEPC Qual tests (code = 3)
        if Broker.CallProc(rpcListLocalLabTests, ['1', '3'], nil, rawData) then
        begin
          rawData.Delete(0);
          if rawData.Count = 0 then
            Result := 1;
        end;
      end;
    end;
  finally
    rawData.Free;
  end;
end;

/////////////////////////////// TFormReportSelector \\\\\\\\\\\\\\\\\\\\\\\\\\\\

procedure TFormReportSelector.acDefaultParamsExecute(Sender: TObject);
begin
  with ReportDescriptor do
    begin
      RestoreDefaultParams;
      ParamsFrame.SetFieldValues(Code, Parameters);
    end;
end;

procedure TFormReportSelector.acLoadParamsExecute(Sender: TObject);
var
  buf: TStringList;
begin
  if Assigned(ReportDescriptor) then
    with FormRPLoadSave do
      if Open(RegInfo, makeInstancePrefix(ReportDescriptor)) = mrOk then
        begin
          buf := TStringList.Create;
          try
            if Broker.CallProc(rpcGUIParameterGet,
              [InstanceName+U+sReportParamsTemplate, Entity],
              nil, buf) then
              begin
                buf.Delete(0);
                // ROR*1.5*19. Fixing ICD-9 parameters format
                buf.Text := FixTemplateFormat(buf.Text,'');
                ReportDescriptor.Parameters.Assign(buf);
                with ReportDescriptor do
                  ParamsFrame.SetFieldValues(Code, Parameters);
              end;
          finally
            buf.Free;
          end;
        end;
end;

// ROR*1.5*19. Testing conversion of the ICD-9 template format.
procedure TFormReportSelector.acLoadTestExecute(Sender: TObject);
var
  buf: TStringList;
begin
  if Assigned(ReportDescriptor) then
    begin
      if dlgOpen.Execute then
        begin
          buf := TStringList.Create;
          buf.LoadFromFile(dlgOpen.FileName);
          buf.Text := FixTemplateFormat(buf.Text,'');

          ReportDescriptor.Parameters.Assign(buf);
          with ReportDescriptor do
            ParamsFrame.SetFieldValues(Code, Parameters);
          buf.Free;
        end;
    end;
end;


procedure TFormReportSelector.acTestUpExecute(Sender: TObject);
begin
{$IFDEF PATCH19TEST}
  frmEventLog.acUp.Execute;
{$ENDIF}
end;

procedure TFormReportSelector.acTestDownExecute(Sender: TObject);
begin
{$IFDEF PATCH19TEST}
  frmEventLog.acDown.Execute;
{$ENDIF}

end;

procedure TFormReportSelector.acTestSaveExecute(Sender: TObject);
var
  buf: TStringList;
begin
  if dlgOpen.Execute then
    begin
      buf := TStringList.Create;
      try
        with ReportDescriptor do
          ParamsFrame.GetFieldValues(Code, Parameters, gfmSave);
        ReportDescriptor.Parameters.AssignTo(buf);
        buf.SaveToFile(dlgOpen.FileName);
      finally
        buf.Free;
      end;
    end;
end;

procedure TFormReportSelector.acSaveParamsExecute(Sender: TObject);
var
  buf: TStringList;
begin
  if Assigned(ReportDescriptor) then
    with FormRPLoadSave do
      if SaveAs(RegInfo, makeInstancePrefix(ReportDescriptor)) = mrOk then
        begin
          Broker.CreateResults;
          buf := TStringList.Create;
          try
            with ReportDescriptor do
              ParamsFrame.GetFieldValues(Code, Parameters, gfmSave);
            ReportDescriptor.Parameters.AssignTo(buf);
{$IFDEF PATCH19TEST}
    if ckbDetails.Checked then
      ShowDebugInfo(buf.Text);
{$ENDIF}
            Broker.CallProc(rpcGUIParameterSet,
              [InstanceName+U+sReportParamsTemplate, Entity], buf);
          finally
            buf.Free;
          end;
        end;
end;

procedure TFormReportSelector.acSelectedUpdate(Sender: TObject);
begin
  TCustomAction(Sender).Enabled := Assigned(lvReports.Selected);
end;

procedure TFormReportSelector.acShowReportListExecute(Sender: TObject);
begin
  pnlReportList.Visible := (chkbShowReportList.Checked);
  splMain.Visible       := (chkbShowReportList.Checked);
end;

procedure TFormReportSelector.ApplyOptions;
var
  ie: TCCRInterfaceElement;
begin
  ie := CCROptions.IE[ieReadOnly];
  lvReports.Color := ie.Color;
  lvReports.Font.Color := ie.FontColor;
end;

procedure TFormReportSelector.btnRunClick(Sender: TObject);
var
  errCtrl: TWinControl;
begin
  ModalResult := mrNone;
  if Assigned(lvReports.Selected) then
    begin
      if Assigned(ReportDescriptor) then
        begin
          with ReportDescriptor do
            errCtrl := ParamsFrame.GetFieldValues(Code, Parameters, gfmRun);
          if Assigned(errCtrl) then
            try
              ActiveControl := errCtrl;
            except
            end
          else
            ModalResult := mrOk;
        end;
    end;
end;

procedure TFormReportSelector.FormCreate(Sender: TObject);
begin
  ApplyOptions;
  //--- Create the report parameters frame
  fParamsFrame := TFrameReportParameters.Create(Self);
  fParamsFrame.Align := alClient;
  fParamsFrame.Parent := pnlReportParams;
{$IFDEF PATCH19TEST}
  pnlDebug.Visible := True;
{$ENDIF}
end;

procedure TFormReportSelector.FormShow(Sender: TObject);
begin
{$IFDEF PATCH19TEST}
  acTestUp.Enabled := Assigned(frmEventLog);
  acTestDown.Enabled := Assigned(frmEventLog);
{$ENDIF}
  //--- Make sure that parameter panels are configured properly
  with lvReports do
    if Assigned(Selected) then
      begin
        //--- Force panel setup for the current report
        ItemFocused := Selected;
        Selected    := nil;
        Selected    := ItemFocused;
      end
    else if Items.Count > 0 then
      begin
        //--- By default, select the first available report from the list
        Selected    := Items[0];
        ItemFocused := Selected;
      end;

  //-- Section 508: Display a warning regarding disabled controls.
  if CCRScreenReaderActive then
    MessageDlg508('', RSC50801, mtWarning, [mbOK], 0);
end;

function TFormReportSelector.getDateTimeToStart: String;
begin
  Result := FloatToStr(WindowsDateTimeToFMDateTime(ParamsFrame.GetDTRun));
end;

function TFormReportSelector.getReportDescriptor: TReportDescriptor;
begin
  if Assigned(lvReports.Selected) then
    Result := lvReports.Selected.Data
  else
    Result := nil;
end;

function TFormReportSelector.getSchedule: String;
begin
  Result := ParamsFrame.cmbbSchedule.Text;
end;

function TFormReportSelector.Load(aRegInfo: TRegistryInfo; Reload: Boolean): TReturnCode;
var
  i, n: Integer;
begin
  Result := RC_OK;
  ParamsFrame.RegInfo := aRegInfo;
  if (aRegInfo <> fRegInfo) or Reload then
  begin
    ParamsFrame.UpdatePanelReferences;
    fRegInfo := aRegInfo;
    n := RegInfo.AvailableReports.Count - 1;
    // Populate the list of reports
    lvReports.Clear;
    for i := 0 to n do
      lvReports.AddItem(RegInfo.AvailableReports[i].Name, RegInfo.AvailableReports[i]);
    for i := lvReports.Items.Count-1 downto 0 do
      lvReports.Items[i].Indent := 0;
    // The list is NOT being sorted when a screen reader is active so we force
    // the reports to sort.
    if CCRScreenReaderActive then
    begin
      lvReports.SortField := -1;
      lvReports.SortField := 0;
    end;
    // Hide all parameter panels
    ParamsFrame.SetupPanels(nil);
  end;
end;

procedure TFormReportSelector.lvReportsEnter(Sender: TObject);
begin
  with lvReports do
    if (Selected = nil) and (ItemFocused <> nil) then
      Selected := ItemFocused;
end;

procedure TFormReportSelector.lvReportsSelectItem(Sender: TObject;
  Item: TListItem; Selected: Boolean);
var
  rpt: TReportDescriptor;
begin
  if not (csDestroying in ComponentState) then
  begin
    rpt := Item.Data;
    if Selected then
    begin
      pnlReportName.Caption := '  ' + rpt.Name;
      Item.ImageIndex := 2;
      ParamsFrame.SetupPanels(rpt);
      DisplayHepCErrorMessage(ParamsFrame.RegInfo.IEN, rpt.Code);
    end
    else begin
      pnlReportName.Caption := '';
      Item.ImageIndex := -1;
      if Assigned(rpt) then
        ParamsFrame.GetFieldValues(rpt.Code, rpt.Parameters);
      ParamsFrame.SetupPanels(nil);
    end;
  end;
end;

function TFormReportSelector.makeInstancePrefix(aReport: TReportDescriptor): String;
begin
  if aReport.SharedTemplates then
    Result := aReport.Code + '::'
  else
    Result := aReport.Code + ':' + RegInfo.Name + ':';
end;

function TFormReportSelector.Open(aRegInfo: TRegistryInfo;
  const aReportCode: String): Boolean;
var rpt: TReportDescriptor;
begin
  Load(aRegInfo);
  fDisplayHepCErrorMsg := true;
  fLastReportCode := '';
  chkbShowReportList.Checked := (aReportCode = '');
  chkbShowReportList.Enabled := (aReportCode = '');
  if aReportCode <> '' then
  begin
    SelectReport(aReportCode);
    Caption := aRegInfo.Description + ' ' + RSC00240;
    //-- Section 508: Add the report name to the screen's caption
    if CCRScreenReaderActive then
    begin
      rpt := getReportDescriptor;
      Caption := Caption + ' - [' + rpt.Name + ']';
    end;
  end
  else
    Caption := aRegInfo.Description + ' ' + RSC00241;
  Result := (ShowModal = mrOk);
end;

function TFormReportSelector.SelectReport(const aReportCode: String): Boolean;
var
  i, n:Integer;
begin
  Result := False;
  with lvReports do
  begin
    n := Items.Count - 1;
    for i := 0 to n do
      if TReportDescriptor(Items[i].Data).Code = aReportCode then
      begin
        Selected := Items[i];
        Result := True;
        Break;
      end;
  end;
end;

procedure TFormReportSelector.vxsFormLayoutBeforeOpen(Sender: TObject);
begin
  vxsFormLayout.Broker := Broker;
end;
{
  The Active property of the vxsComponentState component is set to False so
  that it will not save/restore the component properies automatically.  The
  corresponding save/restore methods are called from the event handlers of
  the vxsFormState component to minimize network data exchange with VistA
  (the VistA storage is opened and closed only once per save/restore process).
}
procedure TFormReportSelector.vxsFormStateRestoreState(Sender: TObject);
begin
  vxsComponentState.RestoreState;
  if Assigned(ParamsFrame) then
    ParamsFrame.vxsComponentState.RestoreState;
end;

procedure TFormReportSelector.vxsFormStateSaveState(Sender: TObject);
begin
  vxsComponentState.SaveState;
  if Assigned(ParamsFrame) then
    ParamsFrame.vxsComponentState.SaveState;
end;

procedure TFormReportSelector.DisplayHepCErrorMessage(aRegistry, aRptCode: String);
var errorMsg: String;
    errorNum: integer;
begin
  errorNum := 0;
  
  //--- Only perform lab test check if this is the HEPC registry and we are
  //--- using the Potential DAA Candidate or Sustained Virologic Response report.
  if (aRegistry = '1') and ((aRptCode = '21') or (aRptCode = '23')) then
  begin
    fDisplayHepCErrorMsg := (fLastReportCode <> aRptCode);
    errorNum := SetErrorCodeForLocalHepCTests;
    if errorNum = 1 then
      errorMsg := RSC00271;
    if fDisplayHepCErrorMsg and (errorNum > 0) then
    begin
      MessageDlg508('', errorMsg, mtError, [mbOK], 0);
      fDisplayHepCErrorMsg := false;
      fLastReportCode := aRptCode;
    end;
  end
  else begin
    fDisplayHepCErrorMsg := true;
    fLastReportCode := aRptCode;
  end;
  btnRun.Enabled := (errorNum = 0);
end;

////////////////////////////////////////////////////////////////////////////////

initialization
finalization
  FormReportSelector := nil;

end.
